<?php
declare(strict_types=1);
session_start();

const DATA_DIR = __DIR__ . '/data';
const UPLOADS_DIR = __DIR__ . '/uploads';

function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }
function json_read(string $path, array $fallback): array {
  if (!file_exists($path)) return $fallback;
  $raw = file_get_contents($path);
  $data = json_decode($raw ?: 'null', true);
  return is_array($data) ? $data : $fallback;
}
function json_write(string $path, array $data): void {
  $dir = dirname($path);
  if (!is_dir($dir)) mkdir($dir, 0775, true);
  file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}
function users_path(): string { return DATA_DIR.'/users.json'; }
function banks_path(string $u): string { return DATA_DIR."/banks_{$u}.json"; }
function config_path(string $u): string { return DATA_DIR."/config_{$u}.json"; }

function bootstrap_demo(): void {
  $u = json_read(users_path(), ['users'=>[]]);
  $found = false;
  foreach ($u['users'] as &$usr){
    if (($usr['username'] ?? '') === 'demo'){
      $found = true;
      if (empty($usr['password_hash'])) $usr['password_hash'] = password_hash('demo', PASSWORD_DEFAULT);
      $usr['is_admin'] = true;
    }
  }
  if (!$found){
    $u['users'][] = ['username'=>'demo','password_hash'=>password_hash('demo', PASSWORD_DEFAULT),'is_admin'=>true];
  }
  json_write(users_path(), $u);

  if (!file_exists(banks_path('demo')) && file_exists(DATA_DIR.'/banks_demo.json')) copy(DATA_DIR.'/banks_demo.json', banks_path('demo'));
  if (!file_exists(config_path('demo')) && file_exists(DATA_DIR.'/config_demo.json')) copy(DATA_DIR.'/config_demo.json', config_path('demo'));
}
bootstrap_demo();

function require_login(): void {
  if (empty($_SESSION['user'])) { header('Location: index.php?page=login'); exit; }
}
function current_user(): ?string { return $_SESSION['user'] ?? null; }

function find_user(string $username): ?array {
  $u = json_read(users_path(), ['users'=>[]]);
  foreach ($u['users'] as $usr){
    if (($usr['username'] ?? '') === $username) return $usr;
  }
  return null;
}
function is_admin(string $username): bool {
  $u = find_user($username);
  return !empty($u['is_admin']);
}
function get_default_banks(): array { return json_read(DATA_DIR.'/banks_demo.json', ['banks'=>[]]); }
function get_default_config(): array { return json_read(DATA_DIR.'/config_demo.json', []); }

function load_user_data(string $u): array {
  $banks = json_read(banks_path($u), get_default_banks());
  $config = json_read(config_path($u), get_default_config());
  if (empty($banks['banks'])) $banks = get_default_banks();
  if (empty($config)) $config = get_default_config();
  return ['banks'=>$banks,'config'=>$config];
}

function body_json(): array {
  $raw = file_get_contents('php://input');
  $data = json_decode($raw ?: 'null', true);
  return is_array($data) ? $data : [];
}
function json_response(array $d): void {
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($d, JSON_UNESCAPED_UNICODE);
  exit;
}
function bank_default_fields(): array {
  return [
    'payee'=>['x_mm'=>20,'y_mm'=>20,'size_pt'=>12,'align'=>'left'],
    'amount_num'=>['x_mm'=>155,'y_mm'=>20,'size_pt'=>12,'align'=>'right'],
    'amount_words'=>['x_mm'=>20,'y_mm'=>36,'size_pt'=>10,'align'=>'left'],
    'city'=>['x_mm'=>120,'y_mm'=>62,'size_pt'=>11,'align'=>'left'],
    'date'=>['x_mm'=>155,'y_mm'=>62,'size_pt'=>11,'align'=>'right'],
  ];
}
function bank_find_index(array $banks, string $id): int {
  foreach ($banks as $i=>$b) if (($b['id'] ?? '') === $id) return $i;
  return -1;
}
function sanitize_bank_id(string $name): string {
  $id = strtolower(trim($name));
  $id = preg_replace('/[^a-z0-9]+/','-',$id);
  $id = trim($id,'-');
  return $id !== '' ? $id : 'bank';
}

$action = $_GET['action'] ?? null;
$page = $_GET['page'] ?? 'saisie';

if ($action){
  require_login();
  $u = current_user(); if (!$u) json_response(['ok'=>false,'error'=>'not logged']);

  if ($action === 'upload_scan'){
    $bankId = $_POST['bank_id'] ?? '';
    if ($bankId==='') json_response(['ok'=>false,'error'=>'bank_id required']);
    if (empty($_FILES['scan'])) json_response(['ok'=>false,'error'=>'no file']);
    $f = $_FILES['scan'];
    if (($f['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) json_response(['ok'=>false,'error'=>'upload error']);
    $ext = strtolower(pathinfo($f['name'] ?? '', PATHINFO_EXTENSION));
    if (!in_array($ext, ['png','jpg','jpeg'], true)) json_response(['ok'=>false,'error'=>'png/jpg uniquement']);
    $ext = $ext === 'jpeg' ? 'jpg' : $ext;

    $dstDir = UPLOADS_DIR . "/{$u}/{$bankId}";
    if (!is_dir($dstDir)) mkdir($dstDir, 0775, true);
    foreach (glob($dstDir.'/scan.*') as $old) @unlink($old);
    $dstName = "scan.$ext";
    $dstPath = $dstDir . "/$dstName";
    move_uploaded_file($f['tmp_name'], $dstPath);

    $ud = load_user_data($u);
    $banks = $ud['banks'];
    $idx = bank_find_index($banks['banks'], $bankId);
    if ($idx < 0) json_response(['ok'=>false,'error'=>'bank not found']);

    $banks['banks'][$idx]['scan_path'] = "uploads/{$u}/{$bankId}/{$dstName}";
    json_write(banks_path($u), $banks);
    json_response(['ok'=>true,'banks'=>$banks]);
  }

  if ($action === 'load'){
    $ud = load_user_data($u);
    json_response(['ok'=>true,'user'=>['username'=>$u,'is_admin'=>is_admin($u)],'banks'=>$ud['banks'],'config'=>$ud['config']]);
  }

  if ($action === 'save'){
    $b = body_json();
    if (!is_array($b['config'] ?? null) || !is_array($b['banks'] ?? null)) json_response(['ok'=>false,'error'=>'payload invalid']);
    json_write(config_path($u), $b['config']);
    json_write(banks_path($u), $b['banks']);
    json_response(['ok'=>true]);
  }

  if ($action === 'reset_bank'){
    $b = body_json();
    $id = (string)($b['bank_id'] ?? '');
    if ($id==='') json_response(['ok'=>false,'error'=>'bank_id required']);
    $ud = load_user_data($u);
    $banks = $ud['banks'];
    $idx = bank_find_index($banks['banks'], $id);
    if ($idx<0) json_response(['ok'=>false,'error'=>'bank not found']);
    $banks['banks'][$idx]['fields'] = bank_default_fields();
    json_write(banks_path($u), $banks);
    json_response(['ok'=>true,'banks'=>$banks]);
  }

  if ($action === 'bank_add'){
    $b = body_json();
    $name = trim((string)($b['name'] ?? ''));
    if ($name==='') json_response(['ok'=>false,'error'=>'name required']);
    $ud = load_user_data($u);
    $banks = $ud['banks'];

    $base = sanitize_bank_id($name);
    $id = $base; $n=2;
    while (bank_find_index($banks['banks'], $id) >= 0){ $id = $base.'-'.$n; $n++; }

    $banks['banks'][] = ['id'=>$id,'name'=>$name,'scan_path'=>'images/blank-cheque-176x80.png','fields'=>bank_default_fields()];
    json_write(banks_path($u), $banks);
    json_response(['ok'=>true,'banks'=>$banks,'new_id'=>$id]);
  }

  if ($action === 'bank_delete'){
    $b = body_json();
    $id = (string)($b['bank_id'] ?? '');
    if ($id==='') json_response(['ok'=>false,'error'=>'bank_id required']);
    $ud = load_user_data($u);
    $banks = $ud['banks'];
    $idx = bank_find_index($banks['banks'], $id);
    if ($idx<0) json_response(['ok'=>false,'error'=>'bank not found']);
    array_splice($banks['banks'], $idx, 1);
    json_write(banks_path($u), $banks);
    json_response(['ok'=>true,'banks'=>$banks]);
  }

  if ($action === 'export'){
    $ud = load_user_data($u);
    $export = ['config'=>$ud['config'], 'banks'=>$ud['banks']];
    $txt = json_encode($export, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="edition_config_'.$u.'.txt"');
    echo $txt; exit;
  }

  if ($action === 'import'){
    $b = body_json();
    $text = (string)($b['text'] ?? '');
    $data = json_decode($text ?: 'null', true);
    if (!is_array($data) || !is_array($data['config'] ?? null) || !is_array($data['banks'] ?? null)){
      json_response(['ok'=>false,'error'=>'JSON invalide (attendu: {config,banks})']);
    }
    json_write(config_path($u), $data['config']);
    json_write(banks_path($u), $data['banks']);
    json_response(['ok'=>true]);
  }

  // users (admin)
  if ($action === 'users_list'){
    if (!is_admin($u)) json_response(['ok'=>false,'error'=>'admin only']);
    $uu = json_read(users_path(), ['users'=>[]]);
    $list = [];
    foreach ($uu['users'] as $usr) $list[] = ['username'=>$usr['username'] ?? '', 'is_admin'=>!empty($usr['is_admin'])];
    json_response(['ok'=>true,'users'=>$list]);
  }
  if ($action === 'users_create'){
    if (!is_admin($u)) json_response(['ok'=>false,'error'=>'admin only']);
    $b = body_json();
    $username = trim((string)($b['username'] ?? ''));
    $password = (string)($b['password'] ?? '');
    $admin = !empty($b['is_admin']);
    if ($username==='' || $password==='') json_response(['ok'=>false,'error'=>'username/password requis']);
    if (!preg_match('/^[a-zA-Z0-9._-]{3,32}$/',$username)) json_response(['ok'=>false,'error'=>'username invalide']);
    $uu = json_read(users_path(), ['users'=>[]]);
    foreach ($uu['users'] as $usr) if (($usr['username'] ?? '') === $username) json_response(['ok'=>false,'error'=>'existe déjà']);
    $uu['users'][] = ['username'=>$username,'password_hash'=>password_hash($password, PASSWORD_DEFAULT),'is_admin'=>$admin];
    json_write(users_path(), $uu);
    if (!file_exists(banks_path($username))) json_write(banks_path($username), get_default_banks());
    if (!file_exists(config_path($username))) json_write(config_path($username), get_default_config());
    json_response(['ok'=>true]);
  }
  if ($action === 'users_delete'){
    if (!is_admin($u)) json_response(['ok'=>false,'error'=>'admin only']);
    $b = body_json();
    $username = trim((string)($b['username'] ?? ''));
    if ($username==='' || $username==='demo') json_response(['ok'=>false,'error'=>'interdit']);
    $uu = json_read(users_path(), ['users'=>[]]);
    $new=[]; $found=false;
    foreach ($uu['users'] as $usr){
      if (($usr['username'] ?? '') === $username){ $found=true; continue; }
      $new[] = $usr;
    }
    if (!$found) json_response(['ok'=>false,'error'=>'introuvable']);
    $uu['users']=$new; json_write(users_path(), $uu);
    @unlink(banks_path($username)); @unlink(config_path($username));
    json_response(['ok'=>true]);
  }

  json_response(['ok'=>false,'error'=>'unknown action']);
}

// Pages
if ($page === 'login'){
  if (!empty($_SESSION['user'])) { header('Location: index.php?page=saisie'); exit; }
  $err = '';
  if ($_SERVER['REQUEST_METHOD'] === 'POST'){
    $username = trim((string)($_POST['username'] ?? ''));
    $password = (string)($_POST['password'] ?? '');
    $usr = find_user($username);
    if ($usr && !empty($usr['password_hash']) && password_verify($password, $usr['password_hash'])){
      $_SESSION['user']=$username;
      header('Location: index.php?page=saisie'); exit;
    } else $err = 'Identifiants invalides';
  }
  ?><!doctype html>
  <html lang="fr"><head>
    <meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">
    <title>EDITION — Login</title>
    <link rel="stylesheet" href="assets/css/app.css">
  </head><body>
    <div class="login-wrap">
      <div class="card" style="width:min(440px,100%)">
        <div class="brand" style="justify-content:center; box-shadow:none">
          <img src="images/logo.svg" alt="EDITION"><div class="title">EDITION</div>
        </div>
        <h1 class="h1" style="text-align:center">Connexion</h1>
        <p class="sub" style="text-align:center">Chèque 176×80 mm + Lettre A4</p>
        <?php if ($err): ?><div class="notice"><strong>Erreur</strong><span><?=h($err)?></span></div><div style="height:12px"></div><?php endif; ?>
        <form method="post" class="stack">
          <div><label>Username</label><input name="username" required></div>
          <div><label>Password</label><input type="password" name="password" required></div>
          <button class="btn primary" type="submit">Se connecter</button>
          <span class="badge">Démo: demo / demo</span>
        </form>
      </div>
    </div>
  </body></html><?php
  exit;
}

if ($page === 'logout'){ session_destroy(); header('Location: index.php?page=login'); exit; }

require_login();
$u = current_user() ?: '';

function nav_link(string $key, string $label, string $ico, string $active): string {
  $href = $key === 'logout' ? 'index.php?page=logout' : 'index.php?page='.$key;
  $cls = $key === $active ? 'active' : '';
  return '<a class="'.$cls.'" href="'.$href.'"><span>'.$ico.'</span><span>'.$label.'</span></a>';
}

?><!doctype html>
<html lang="fr"><head>
  <meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">
  <title>EDITION</title>
  <link rel="stylesheet" href="assets/css/app.css">
</head><body>
<div class="layout" id="app">
  <aside class="sidebar">
    <div class="brand">
      <img src="images/logo.svg" alt="EDITION">
      <div><div class="title">EDITION</div><div style="font-size:12px;color:var(--muted)"><?=h($u)?></div></div>
    </div>
    <nav class="nav">
      <?=nav_link('saisie','Saisie','✍️',$page)?>
      <?=nav_link('settings','Réglages','⚙️',$page)?>
      <?=nav_link('preview','Aperçu PDF chèque','🖨️',$page)?>
      <?=nav_link('letter','Lettre','📄',$page)?>
      <?=nav_link('users','Utilisateurs','👤',$page)?>
      <?=nav_link('logout','Déconnexion','⎋',$page)?>
    </nav>
    <div style="margin-top:16px"><span class="badge ok" id="dirtyBadge">Tout est enregistré</span></div>
  </aside>

  <main class="main">
    <div class="topbar">
      <div>
        <h1 class="h1"><?=h($page==='saisie'?'Saisie':($page==='settings'?'Réglages':($page==='preview'?'Aperçu PDF chèque':($page==='letter'?'Lettre':($page==='users'?'Utilisateurs':'EDITION')))))?></h1>
        <p class="sub">PDF texte Helvetica (sans scan)</p>
      </div>
      <div class="toolbar">
        <button class="btn primary" id="saveBtn">💾 Enregistrer</button>
        <?php if ($page==='saisie' || $page==='preview'): ?><button class="btn ok" id="printChequeBtn">🖨️ Imprimer chèque</button><?php endif; ?>
        <?php if ($page==='saisie' || $page==='letter'): ?><button class="btn ok" id="printLetterBtn">📄 Imprimer lettre</button><?php endif; ?>
        <span class="spacer"></span>
        <button class="btn" id="exportBtn">⬇️ Export config</button>
        <button class="btn" id="importBtn">⬆️ Import config</button>
        <input type="file" id="importFile" accept=".txt,application/json,text/plain" style="display:none">
      </div>
    </div>

    <div id="toast" class="notice hidden"></div>

    <?php if ($page==='saisie'): ?>
      <div class="grid2">
        <section class="card">
          <h3>Informations</h3>
          <div class="row">
            <div><label>Banque</label><select data-bank-select></select></div>
            <div><label>Date</label><input id="date" placeholder="JJ/MM/AAAA"></div>
          </div>
          <div class="row">
            <div><label>Ville</label><input id="city" placeholder="Casablanca"></div>
            <div><label>Bénéficiaire</label><input id="payee" placeholder="Nom du bénéficiaire"></div>
          </div>
          <div class="row">
            <div><label>Montant (chiffres)</label><input id="amount_num" inputmode="decimal" placeholder="0.00"></div>
            <div><label>Montant (lettres) (auto)</label><input id="amount_words" readonly></div>
          </div>
          <hr>
          <div class="row">
            <div><label>N° chèque (lettre uniquement)</label><input id="cheque_no" placeholder="000000"></div>
            <div><label>Note (lettre uniquement)</label><input id="note" placeholder="Optionnel"></div>
          </div>
          <div>
            <label>Factures payées (1 par ligne)</label>
            <textarea id="invoices" placeholder="FAC001&#10;FAC002"></textarea>
            <small>Lettre limitée à 1 page (factures en trop ignorées).</small>
          </div>
        </section>

        <section class="card preview-root">
          <h3>Prévisualisation chèque (guide)</h3>
          <div class="scanInfo" style="color:var(--muted); font-size:12px; margin-bottom:10px"></div>
          <div class="notice ratioNotice hidden"><strong>Attention</strong><span>Scan hors format (176×80)</span></div>
          <div class="preview-wrap">
            <img class="preview-img" src="images/blank-cheque-176x80.png" alt="scan">
            <div class="tag" data-field-tag="payee" data-label="Bénéficiaire">Bénéficiaire</div>
            <div class="tag" data-field-tag="amount_num" data-label="Montant (chiffres)">Montant (chiffres)</div>
            <div class="tag" data-field-tag="amount_words" data-label="Montant (lettres)">Montant (lettres)</div>
            <div class="tag" data-field-tag="city" data-label="Ville">Ville</div>
            <div class="tag" data-field-tag="date" data-label="Date">Date</div>
          </div>
          <small style="display:block;margin-top:10px;color:var(--muted)">Le scan n’est jamais imprimé dans le PDF.</small>
        </section>
      </div>

    <?php elseif ($page==='settings'): ?>
      <div class="grid2">
        <section class="card preview-root">
          <h3>Banques</h3>
          <div class="row">
            <div><label>Banque</label><select data-bank-select></select></div>
            <div class="toolbar" style="align-self:end">
              <button class="btn" id="addBankBtn">➕ Ajouter banque</button>
              <button class="btn danger" id="delBankBtn">🗑️ Supprimer banque</button>
              <button class="btn" id="resetBankBtn">↩️ Réinitialiser banque</button>
            </div>
          </div>
          <hr>
          <h3>Scan banque (guide)</h3>
          <div class="row">
            <div>
              <label>Importer scan (png/jpg)</label>
              <input id="scanFile" type="file" accept="image/png,image/jpeg">
              <small>/uploads/&lt;user&gt;/&lt;bankId&gt;/scan.*</small>
            </div>
            <div>
              <label>État scan</label>
              <div class="scanInfo" style="color:var(--muted); font-size:12px"></div>
              <div class="notice ratioNotice hidden" style="margin-top:10px"><strong>Attention</strong><span>Scan hors format (176×80)</span></div>
            </div>
          </div>
          <hr>
          <h3>Prévisualisation</h3>
          <div class="preview-wrap">
            <img class="preview-img" src="images/blank-cheque-176x80.png" alt="scan">
            <div class="tag" data-field-tag="payee" data-label="Bénéficiaire">Bénéficiaire</div>
            <div class="tag" data-field-tag="amount_num" data-label="Montant (chiffres)">Montant (chiffres)</div>
            <div class="tag" data-field-tag="amount_words" data-label="Montant (lettres)">Montant (lettres)</div>
            <div class="tag" data-field-tag="city" data-label="Ville">Ville</div>
            <div class="tag" data-field-tag="date" data-label="Date">Date</div>
          </div>
        </section>

        <section class="card">
          <h3>Champs imprimables (positions)</h3>
          <div class="table-grid" id="fieldsGrid"></div>
          <small>Positions en mm. Tags draggable synchronisent ce tableau.</small>
        </section>
      </div>

    <?php elseif ($page==='preview'): ?>
      <div class="card">
        <h3>Aperçu PDF chèque (sans scan)</h3>
        <div class="pdf-frame" id="pdfHolder"></div>
        <div style="height:12px"></div>
        <button class="btn ok" id="printPreviewBtn">🖨️ Imprimer</button>
      </div>

    <?php elseif ($page==='letter'): ?>
      <div class="grid2">
        <section class="card">
          <h3>Paramètres lettre</h3>
          <div class="row">
            <div><label>Nom entreprise</label><input id="company_name" placeholder="Votre Entreprise"></div>
            <div><label>Signataire</label><input id="signatory" placeholder="Signature"></div>
          </div>
          <div style="height:12px"></div>
          <div>
            <label>Contenu TEMPLATE (optionnel)</label>
            <textarea id="letter_template" placeholder="Contenu (automatique)"></textarea>
            <small>Variables: {CITY} {DATE} {CHEQUE_NO} {PAYEE} {AMOUNT_NUM} {AMOUNT_WORDS} {INVOICES}</small>
          </div>
          <hr>
          <h3>Contenu (automatique)</h3>
          <p style="color:var(--muted);margin:0">Si le template est vide, l’app génère la lettre automatiquement.</p>
          <div style="height:12px"></div>
          <button class="btn ok" id="printLetterPageBtn">📄 Imprimer lettre</button>
        </section>
        <section class="card">
          <h3>Rappel</h3>
          <p style="color:var(--muted);margin:0">Remplis dans “Saisie” : date, ville, n° chèque, bénéficiaire, montant, factures.</p>
        </section>
      </div>

    <?php elseif ($page==='users'): ?>
      <?php if (!is_admin($u)): ?>
        <div class="card"><h3>Accès refusé</h3><p class="sub">Admin uniquement.</p></div>
      <?php else: ?>
        <div class="grid2">
          <section class="card">
            <h3>Créer utilisateur</h3>
            <form id="userCreateForm" class="stack">
              <div class="row">
                <div><label>Username</label><input id="new_username"></div>
                <div><label>Password</label><input id="new_password" type="password"></div>
              </div>
              <label style="display:flex; gap:10px; align-items:center">
                <input id="new_admin" type="checkbox" style="width:auto"><span style="color:var(--muted);font-size:12px">Admin</span>
              </label>
              <button class="btn primary" type="submit">➕ Créer</button>
              <small>Passwords via <code>password_hash()</code>.</small>
            </form>
          </section>
          <section class="card">
            <h3>Utilisateurs</h3>
            <div style="overflow:auto">
              <table style="width:100%; border-collapse:collapse">
                <thead><tr>
                  <th style="text-align:left;padding:10px;border-bottom:1px solid var(--border);color:var(--muted);font-size:12px">Username</th>
                  <th style="text-align:left;padding:10px;border-bottom:1px solid var(--border);color:var(--muted);font-size:12px">Rôle</th>
                  <th style="text-align:right;padding:10px;border-bottom:1px solid var(--border);color:var(--muted);font-size:12px">Action</th>
                </tr></thead>
                <tbody id="usersTable"></tbody>
              </table>
            </div>
          </section>
        </div>
      <?php endif; ?>
    <?php endif; ?>

  </main>
</div>

<script src="assets/js/pdf.js"></script>
<script src="assets/js/app.js"></script>
<script>
(function(){
  // Users page JS (small, isolated)
  const $ = (s)=>document.querySelector(s);
  async function api(action, payload){
    const r = await fetch(`index.php?action=${encodeURIComponent(action)}`,{
      method:'POST', headers:{'Content-Type':'application/json'},
      body: JSON.stringify(payload||{})
    });
    return await r.json();
  }
  const table = $('#usersTable');
  const form = $('#userCreateForm');
  if (table){
    api('users_list',{}).then(r=>{
      if (!r.ok) return;
      table.innerHTML = r.users.map(u=>`
        <tr>
          <td style="padding:10px;border-bottom:1px solid rgba(148,163,184,.18)">${u.username}</td>
          <td style="padding:10px;border-bottom:1px solid rgba(148,163,184,.18)">${u.is_admin?'admin':'user'}</td>
          <td style="padding:10px;border-bottom:1px solid rgba(148,163,184,.18); text-align:right">
            ${u.username==='demo'?'':'<button class="btn danger" data-del="'+u.username+'">Supprimer</button>'}
          </td>
        </tr>
      `).join('');
      document.querySelectorAll('[data-del]').forEach(btn=>{
        btn.addEventListener('click', async ()=>{
          const username = btn.getAttribute('data-del');
          const rr = await api('users_delete',{username});
          if (rr.ok) location.reload();
          else alert(rr.error||'Erreur');
        });
      });
    });
  }
  if (form){
    form.addEventListener('submit', async (e)=>{
      e.preventDefault();
      const u = $('#new_username').value.trim();
      const p = $('#new_password').value;
      const a = $('#new_admin').checked;
      const r = await api('users_create',{username:u,password:p,is_admin:a});
      if (r.ok) location.reload();
      else alert(r.error||'Erreur');
    });
  }
})();
</script>
</body></html>
