/*
  EDITION PDF engine (no deps)
  - Single page
  - Base font: Helvetica (built-in)
  - Text only (no scan)
  - Supports WinAnsi encoding via octal escapes for non-ASCII bytes
  - Coordinates: input in mm from TOP-LEFT; PDF uses pt from BOTTOM-LEFT.
*/
(function(){
  'use strict';

  const PT_PER_MM = 72 / 25.4;

  function mmToPt(mm){ return (Number(mm)||0) * PT_PER_MM; }

  function toWinAnsiByte(ch){
    const cp = ch.codePointAt(0);

    // Normalize common typography to ASCII
    if (cp === 0x2019 || cp === 0x2018) return 0x27; // ’ ‘ -> '
    if (cp === 0x201C || cp === 0x201D) return 0x22; // “ ” -> "
    if (cp === 0x2013 || cp === 0x2014) return 0x2D; // – — -> -
    if (cp === 0x00A0) return 0x20; // nbsp
    if (cp === 0x2026) return 0x2E; // …

    // Latin-1 bytes map directly
    if (cp >= 0x00 && cp <= 0xFF) return cp;

    // Fallback
    return 0x3F; // ?
  }

  function escPdfLiteralWinAnsi(str){
    const s = String(str ?? '');
    let out = '';
    for (const ch of s){
      const b = toWinAnsiByte(ch) & 0xFF;

      // Escape PDF special chars in literal strings
      if (b === 0x5C) { out += '\\\\'; continue; } // \
      if (b === 0x28) { out += '\\\\(' ; continue; } // (
      if (b === 0x29) { out += '\\\\)' ; continue; } // )

      if (b >= 0x20 && b <= 0x7E){
        out += String.fromCharCode(b);
      } else {
        // octal escape: \ddd
        const oct = b.toString(8).padStart(3,'0');
        out += '\\\\' + oct;
      }
    }
    return out;
  }

  // rough width for alignment (Helvetica heuristic)
  function approxTextWidthPt(text, sizePt){
    const s = String(text ?? '');
    let units = 0;
    for (let i=0;i<s.length;i++){
      const c = s[i];
      if (c === ' ') units += 0.28;
      else if ('ilI.,:;!|'.includes(c)) units += 0.28;
      else if ('mwMW'.includes(c)) units += 0.78;
      else units += 0.52;
    }
    return units * (Number(sizePt)||12);
  }

  function buildContent({pageWmm, pageHmm, items}){
    const pageWpt = mmToPt(pageWmm);
    const pageHpt = mmToPt(pageHmm);
    const out = [];
    out.push('BT');
    out.push('/F1 12 Tf');

    (items || []).forEach(it=>{
      if (!it) return;
      const text = String(it.text ?? '');
      const size = Number(it.size_pt ?? 12) || 12;
      const xMm = Number(it.x_mm ?? 0) || 0;
      const yTopMm = Number(it.y_mm ?? 0) || 0;
      const align = String(it.align ?? 'left').toLowerCase();

      let xPt = mmToPt(xMm);
      const yPt = pageHpt - mmToPt(yTopMm);

      if (align === 'center' || align === 'right'){
        const w = approxTextWidthPt(text, size);
        if (align === 'center') xPt -= (w/2);
        else xPt -= w;
      }

      out.push(`/F1 ${size.toFixed(2)} Tf`);
      // absolute placement matrix (prevents cumulative offsets)
      out.push(`1 0 0 1 ${xPt.toFixed(2)} ${yPt.toFixed(2)} Tm`);
      out.push(`(${escPdfLiteralWinAnsi(text)}) Tj`);
    });

    out.push('ET');
    return out.join('\n');
  }

  function makeSinglePagePdfBlob({pageWmm, pageHmm, items}){
    const content = buildContent({pageWmm, pageHmm, items});
    const enc = new TextEncoder();
    const contentBytes = enc.encode(content);
    const contentLen = contentBytes.length;

    const pageWpt = mmToPt(pageWmm);
    const pageHpt = mmToPt(pageHmm);

    const chunks = [];
    function pushStr(s){ chunks.push(enc.encode(s)); }
    function sizeSoFar(){ return sumLen(chunks); }
    function sumLen(arr){ let n=0; for (const a of arr) n += a.length; return n; }

    pushStr('%PDF-1.4\n%\xE2\xE3\xCF\xD3\n');

    // object offsets
    const xref = [];
    xref[0] = 0;

    function startObj(n){
      xref[n] = sizeSoFar();
      pushStr(`${n} 0 obj\n`);
    }
    function endObj(){ pushStr('endobj\n'); }

    // 1 Catalog
    startObj(1);
    pushStr('<< /Type /Catalog /Pages 2 0 R >>\n');
    endObj();

    // 2 Pages
    startObj(2);
    pushStr('<< /Type /Pages /Kids [3 0 R] /Count 1 >>\n');
    endObj();

    // 5 Font Helvetica
    startObj(5);
    pushStr('<< /Type /Font /Subtype /Type1 /BaseFont /Helvetica >>\n');
    endObj();

    // 3 Page
    startObj(3);
    pushStr('<< /Type /Page /Parent 2 0 R ');
    pushStr(`/MediaBox [0 0 ${pageWpt.toFixed(2)} ${pageHpt.toFixed(2)}] `);
    pushStr('/Resources << /Font << /F1 5 0 R >> >> ');
    pushStr('/Contents 4 0 R >>\n');
    endObj();

    // 4 Content stream
    startObj(4);
    pushStr(`<< /Length ${contentLen} >>\nstream\n`);
    chunks.push(contentBytes);
    pushStr('\nendstream\n');
    endObj();

    // xref table
    const xrefStart = sizeSoFar();
    pushStr('xref\n');
    const size = 6; // 0..5
    pushStr(`0 ${size}\n`);
    pushStr('0000000000 65535 f \n');
    for (let i=1;i<size;i++){
      const off = String(xref[i] || 0).padStart(10,'0');
      pushStr(`${off} 00000 n \n`);
    }

    // trailer
    pushStr('trailer\n');
    pushStr(`<< /Size ${size} /Root 1 0 R >>\n`);
    pushStr('startxref\n');
    pushStr(`${xrefStart}\n`);
    pushStr('%%EOF');

    return new Blob(chunks, {type:'application/pdf'});
  }

  function openPdfInNewTab(blob){
    const url = URL.createObjectURL(blob);
    window.open(url, '_blank');
    return url;
  }

  function openAndPrintPdf(blob){
    const url = URL.createObjectURL(blob);
    const w = window.open(url, '_blank');
    if (!w) return url;
    // Best-effort print after load
    const timer = setInterval(()=>{
      try{
        if (w.document && w.document.readyState === 'complete'){
          clearInterval(timer);
          w.focus();
          w.print();
        }
      }catch(e){}
    }, 300);
    return url;
  }

  window.EDITION_PDF = { mmToPt, makeSinglePagePdfBlob, openAndPrintPdf, openPdfInNewTab };
})();
