(function(){
  'use strict';

  const CHECK_W_MM = 176;
  const CHECK_H_MM = 80;
  const RATIO = CHECK_W_MM / CHECK_H_MM;
  const EPS = 0.01;

  const $ = (s, r=document)=>r.querySelector(s);
  const $$ = (s, r=document)=>Array.from(r.querySelectorAll(s));

  const state = {
    user:null,
    banks:[],
    config:null,
    currentBankId:null,
    dirty:false,
    drag:{active:false, field:null, startX:0, startY:0, startL:0, startT:0}
  };

  function todayDdMmYyyy(){
    const d = new Date();
    const dd = String(d.getDate()).padStart(2,'0');
    const mm = String(d.getMonth()+1).padStart(2,'0');
    const yy = String(d.getFullYear());
    return `${dd}/${mm}/${yy}`;
  }

  function getPdfApi(){
    const api = window.EDITION_PDF;
    if (!api || typeof api.makeSinglePagePdfBlob !== 'function'){
      console.error('EDITION_PDF missing or invalid', api);
      throw new Error('Moteur PDF non chargé (pdf.js).');
    }
    return api;
  }

  function toast(msg){
    const t = $('#toast');
    if (!t) { alert(msg); return; }
    t.textContent = msg;
    t.classList.remove('hidden');
    clearTimeout(toast._t);
    toast._t = setTimeout(()=>t.classList.add('hidden'), 2500);
  }

  async function api(action, payload){
    const r = await fetch(`index.php?action=${encodeURIComponent(action)}`,{
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify(payload||{})
    });
    return await r.json();
  }

  function setDirty(v){
    state.dirty = !!v;
    const b = $('#dirtyBadge');
    if (!b) return;
    b.classList.toggle('dirty', state.dirty);
    b.classList.toggle('ok', !state.dirty);
    b.textContent = state.dirty ? 'Modifications non enregistrées' : 'Tout est enregistré';
  }

  // --- mm <-> px (single source of truth)
  function metrics(previewEl){
    if (!previewEl) return null;
    const rect = previewEl.getBoundingClientRect();
    return {w:rect.width, h:rect.height};
  }
  function mmToPx(xMm, yMm, previewEl){
    const m = metrics(previewEl); if (!m) return {x:0,y:0};
    return { x: (xMm/CHECK_W_MM)*m.w, y: (yMm/CHECK_H_MM)*m.h };
  }
  function pxToMm(xPx, yPx, previewEl){
    const m = metrics(previewEl); if (!m) return {x_mm:0,y_mm:0};
    return { x_mm:(xPx/m.w)*CHECK_W_MM, y_mm:(yPx/m.h)*CHECK_H_MM };
  }

  function getBank(){
    return state.banks.find(b=>b.id===state.currentBankId) || state.banks[0] || null;
  }

  function fillBankSelects(){
    $$('[data-bank-select]').forEach(sel=>{
      sel.innerHTML = state.banks.map(b=>`<option value="${b.id}">${escapeHtml(b.name)}</option>`).join('');
      sel.value = state.currentBankId || (state.banks[0]?.id||'');
      sel.onchange = ()=>{
        state.currentBankId = sel.value;
        state.config.last_bank_id = state.currentBankId;
        setDirty(true);
        fillBankSelects();
        renderAll();
      };
    });
  }

  function escapeHtml(s){
    return String(s??'').replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;').replace(/'/g,'&#039;');
  }

  function fmtMoney(v){
    // Accept comma or dot as decimal separator. Output: "123,40" (always 2 decimals)
    const s = String(v ?? '').trim();
    if (!s) return '';
    const norm = s.replace(',', '.');
    const cleaned = norm.replace(/[^\d.]/g,'');
    if (!cleaned) return '';
    const parts = cleaned.split('.');
    const i = parseInt(parts[0] || '0', 10);
    let d = (parts[1] || '');
    d = (d + '00').slice(0,2); // force 2 decimals
    return `${i},${d}`;
  }



  // French words
  function numberToFrenchWords(n){
    n = Number(n);
    if (!Number.isFinite(n) || n<0) return '';
    if (n===0) return 'zéro';
    const u=['','un','deux','trois','quatre','cinq','six','sept','huit','neuf'];
    const t=['dix','onze','douze','treize','quatorze','quinze','seize','dix-sept','dix-huit','dix-neuf'];
    const d=['','dix','vingt','trente','quarante','cinquante','soixante','soixante','quatre-vingt','quatre-vingt'];

    function under100(x){
      if (x<10) return u[x];
      if (x<20) return t[x-10];
      const ten=Math.floor(x/10), one=x%10;
      if (ten===7 || ten===9){
        const base = ten===7 ? 'soixante' : 'quatre-vingt';
        const rest = x - (ten===7 ? 60 : 80);
        if (ten===7 && rest===11) return `${base} et onze`;
        return `${base}-${under100(rest)}`;
      }
      if (ten===8){
        if (one===0) return 'quatre-vingts';
        return `quatre-vingt-${u[one]}`;
      }
      if (one===0) return d[ten];
      if (one===1 && [2,3,4,5,6].includes(ten)) return `${d[ten]} et un`;
      return `${d[ten]}-${u[one]}`;
    }
    function under1000(x){
      if (x<100) return under100(x);
      const h=Math.floor(x/100), rest=x%100;
      let out = h===1 ? 'cent' : `${u[h]} cent`;
      if (rest===0 && h>1) out+='s';
      if (rest>0) out += ` ${under100(rest)}`;
      return out;
    }

    let v=Math.floor(n);
    const parts=[];
    const scales=[
      [1_000_000_000,'milliard','milliards'],
      [1_000_000,'million','millions'],
      [1000,'mille','mille']
    ];
    for (const [scale,sing,plur] of scales){
      const c=Math.floor(v/scale);
      if (c){
        if (scale===1000){
          parts.push(c===1?'mille':`${under1000(c)} mille`);
        }else{
          parts.push(c===1?`un ${sing}`:`${under1000(c)} ${plur}`);
        }
        v = v % scale;
      }
    }
    if (v) parts.push(under1000(v));
    return parts.join(' ');
  }

  function wordsMad(input){
    const s=String(input??'').replace(',', '.').trim();
    if (!s) return '';
    const cleaned=s.replace(/[^\d.]/g,'');
    if (!cleaned) return '';
    const p=cleaned.split('.');
    const dir=parseInt(p[0]||'0',10);
    const cent=parseInt((p[1]||'').padEnd(2,'0').slice(0,2)||'0',10);
    let out = `${numberToFrenchWords(dir)} dirhams`;
    if (cent>0) out += ` et ${numberToFrenchWords(cent)} centimes`;
    // Capitalize first letter
    out = out.charAt(0).toUpperCase() + out.slice(1);

    return out;
  }

  function bindInputs(){
    const c = state.config.check;
    const bind = (id, fn)=>{
      const el = $(id);
      if (!el) return;
      el.addEventListener('input', fn);
    };
    bind('#date', ()=>{ c.date = $('#date').value; setDirty(true); updateTagsText(); });
    bind('#city', ()=>{ c.city = $('#city').value; setDirty(true); updateTagsText(); });
    bind('#payee', ()=>{ c.payee = $('#payee').value; setDirty(true); updateTagsText(); });
    (function(){
      const el = $('#amount_num');
      if (!el) return;

      const updateFrom = (val)=>{
        c.amount_num = val;
        c.amount_words = wordsMad(val);
        const aw = $('#amount_words'); if (aw) aw.value = c.amount_words;
        setDirty(true); updateTagsText();
      };

      const sanitizeDraft = (raw)=>{
        let s = String(raw ?? '').replace(/[^\d.,]/g,'');
        const m = s.match(/[\.,]/);
        if (!m) return s;
        const idx = s.indexOf(m[0]);
        const before = s.slice(0, idx).replace(/[\.,]/g,'');
        const after = s.slice(idx+1).replace(/[\.,]/g,'');
        const hadTrailingSep = idx === s.length - 1;
        let out = before + ',' + after;
        if (hadTrailingSep && after.length === 0) out = before + ','; // keep comma while typing
        return out;
      };

      el.addEventListener('input', ()=>{
        const draft = sanitizeDraft(el.value);
        el.value = draft;
        updateFrom(draft);
      });

      el.addEventListener('blur', ()=>{
        const finalVal = fmtMoney(el.value);
        el.value = finalVal;
        updateFrom(finalVal);
      });
    })();
    bind('#cheque_no', ()=>{ c.cheque_no = $('#cheque_no').value; setDirty(true); });
    bind('#note', ()=>{ c.note = $('#note').value; setDirty(true); });
    const inv = $('#invoices');
    if (inv){
      inv.addEventListener('input', ()=>{
        c.invoices = inv.value.split(/\r?\n/).map(s=>s.trim()).filter(Boolean);
        setDirty(true);
      });
    }

    const company = $('#company_name');
    if (company){
      company.addEventListener('input', ()=>{
        state.config.letter.company_name = company.value;
        setDirty(true);
      });
    }
    const sign = $('#signatory');
    if (sign){
      sign.addEventListener('input', ()=>{
        state.config.letter.signatory = sign.value;
        setDirty(true);
      });
    }
    const tpl = $('#letter_template');
    if (tpl){
      tpl.addEventListener('input', ()=>{
        state.config.letter.template = tpl.value;
        setDirty(true);
      });
    }
  }

  function loadForm(){
    const c = state.config.check;
    // default date today
    if (!c.date) c.date = todayDdMmYyyy();

    const set = (id, v)=>{ const el=$(id); if (el) el.value = v??''; };
    set('#date', c.date);
    set('#city', c.city);
    set('#payee', c.payee);
    set('#amount_num', c.amount_num);
    set('#amount_words', c.amount_words);
    set('#cheque_no', c.cheque_no);
    set('#note', c.note);
    set('#invoices', (c.invoices||[]).join('\n'));

    set('#company_name', state.config.letter.company_name);
    set('#signatory', state.config.letter.signatory);
    const tpl = $('#letter_template');
    if (tpl){
      tpl.value = state.config.letter.template || '';
      tpl.placeholder = 'Contenu (automatique)';
    }
  }

  function renderPreview(rootEl){
    const bank = getBank();
    const img = $('.preview-img', rootEl);
    const warn = $('.ratioNotice', rootEl);
    const info = $('.scanInfo', rootEl);
    if (img){
      img.src = bank?.scan_path || 'images/blank-cheque-176x80.png';
      img.onload = ()=>{
        const ratio = img.naturalWidth && img.naturalHeight ? img.naturalWidth/img.naturalHeight : RATIO;
        const ok = Math.abs(ratio - RATIO) <= (RATIO*EPS);
        if (warn) warn.classList.toggle('hidden', ok);
        if (info){
          info.innerHTML = `Scan: ${img.naturalWidth||'?'}×${img.naturalHeight||'?'} px — Ratio: ${ratio.toFixed(3)} <span class="badge ${ok?'ok':''}">${ok?'OK':'Scan hors format (176×80)'}</span>`;
        }
      };
      img.onerror = ()=>{ img.src = 'images/blank-cheque-176x80.png'; };
    }
    positionTags(rootEl);
    updateTagsText();
  }

  function positionTags(rootEl){
    const bank = getBank();
    const wrap = $('.preview-wrap', rootEl);
    if (!bank || !wrap) return;
    const fields = bank.fields || {};
    $$('[data-field-tag]', rootEl).forEach(tag=>{
      const k = tag.dataset.fieldTag;
      const f = fields[k]; if (!f) return;
      const p = mmToPx(Number(f.x_mm)||0, Number(f.y_mm)||0, wrap);
      tag.style.left = `${p.x}px`;
      tag.style.top  = `${p.y}px`;
    });
  }

  function updateTagsText(){
    const c = state.config.check;
    const map = {
      payee: c.payee || 'Bénéficiaire',
      amount_num: c.amount_num ? `#${c.amount_num}#` : 'Montant (chiffres)',
      amount_words: c.amount_words || 'Montant (lettres)',
      city: c.city || 'Ville',
      date: c.date || 'Date'
    };
    $$('.tag[data-field-tag]').forEach(tag=>{
      const k = tag.dataset.fieldTag;
      const val = map[k] || tag.dataset.label || k;
      tag.textContent = val;
      const empty = !String(c[k]||'').trim();
      tag.classList.toggle('empty', empty);
    });
  }

  function enableDragging(rootEl){
    const wrap = $('.preview-wrap', rootEl);
    if (!wrap) return;
    $$('[data-field-tag]', rootEl).forEach(tag=>{
      tag.onpointerdown = (e)=>{
        tag.setPointerCapture(e.pointerId);
        state.drag.active = true;
        state.drag.field = tag.dataset.fieldTag;
        state.drag.startX = e.clientX;
        state.drag.startY = e.clientY;
        state.drag.startL = parseFloat(tag.style.left||'0');
        state.drag.startT = parseFloat(tag.style.top||'0');
      };
      tag.onpointermove = (e)=>{
        if (!state.drag.active) return;
        const bank = getBank(); if (!bank) return;
        const field = state.drag.field;
        const f = bank.fields?.[field]; if (!f) return;

        const dx = e.clientX - state.drag.startX;
        const dy = e.clientY - state.drag.startY;
        const newL = Math.max(0, Math.min(state.drag.startL + dx, wrap.clientWidth));
        const newT = Math.max(0, Math.min(state.drag.startT + dy, wrap.clientHeight));
        tag.style.left = `${newL}px`;
        tag.style.top  = `${newT}px`;

        const mm = pxToMm(newL, newT, wrap);
        f.x_mm = Math.round(mm.x_mm*100)/100;
        f.y_mm = Math.round(mm.y_mm*100)/100;
        syncSettingsGrid();
        setDirty(true);
      };
      tag.onpointerup = ()=>{ state.drag.active=false; };
      tag.onpointercancel = ()=>{ state.drag.active=false; };
    });
  }

  function renderSettingsGrid(){
    const grid = $('#fieldsGrid');
    if (!grid) return;
    const bank = getBank(); if (!bank) return;

    const labels = {
      payee:'Bénéficiaire',
      amount_num:'Montant (chiffres)',
      amount_words:'Montant (lettres)',
      city:'Ville',
      date:'Date'
    };

    grid.innerHTML = '';
    Object.keys(labels).forEach(k=>{
      const f = bank.fields[k];
      const box = document.createElement('div');
      box.className = 'fieldbox';
      box.innerHTML = `
        <div class="meta"><div><div class="name">${labels[k]}</div><div class="code">${k}</div></div></div>
        <div class="controls">
          <div><label>X (mm)</label><input data-f="${k}" data-p="x_mm" type="number" step="0.1" value="${f.x_mm}"></div>
          <div><label>Y (mm)</label><input data-f="${k}" data-p="y_mm" type="number" step="0.1" value="${f.y_mm}"></div>
          <div><label>Taille (pt)</label><input data-f="${k}" data-p="size_pt" type="number" step="0.5" value="${f.size_pt}"></div>
          <div><label>Align</label>
            <select data-f="${k}" data-p="align">
              ${['left','center','right'].map(a=>`<option value="${a}" ${f.align===a?'selected':''}>${a}</option>`).join('')}
            </select>
          </div>
        </div>
      `;
      grid.appendChild(box);
    });

    $$('[data-f]', grid).forEach(el=>{
      el.oninput = ()=>{
        const bank = getBank(); if (!bank) return;
        const k = el.dataset.f, p = el.dataset.p;
        const f = bank.fields[k];
        f[p] = (el.tagName==='SELECT') ? el.value : Number(el.value);
        setDirty(true);
        renderAllPreviews();
      };
    });
  }

  function syncSettingsGrid(){
    const grid = $('#fieldsGrid');
    if (!grid) return;
    const bank = getBank(); if (!bank) return;
    $$('[data-f]', grid).forEach(el=>{
      const k=el.dataset.f, p=el.dataset.p;
      const val = bank.fields[k][p];
      if (el.tagName==='SELECT') el.value = val;
      else el.value = val;
    });
  }

  function buildChequeItems(){
    const bank = getBank();
    const c = state.config.check;
    const f = bank.fields;
    return [
      {...f.payee, text:c.payee||''},
      {...f.amount_num, text:(c.amount_num?`#${c.amount_num}#`:'')},
      {...f.amount_words, text:c.amount_words||''},
      {...f.city, text:c.city||''},
      {...f.date, text:c.date||''},
    ];
  }

  function applyTemplate(tpl, vars){
    const raw = String(tpl||'').trim();
    if (!raw) return null;
    let out = raw;
    for (const k in vars){
      out = out.replace(new RegExp(`\\{${k}\\}`,'g'), String(vars[k]??''));
    }
    return out.split(/\r?\n/);
  }

  function buildLetterLines(){
    const c = state.config.check;
    const L = state.config.letter;
    const city = c.city || 'Ville';
    const date = c.date || '';
    const chequeNo = c.cheque_no || '';
    const payee = c.payee || '';
    const amountNum = c.amount_num || '';
    const amountWords = c.amount_words || '';
    const invoices = (c.invoices||[]).slice(0, 18);

    const vars = {
      CITY: city,
      DATE: date,
      CHEQUE_NO: chequeNo,
      PAYEE: payee,
      AMOUNT_NUM: amountNum,
      AMOUNT_WORDS: amountWords,
      INVOICES: invoices.join('\n')
    };
    const templated = applyTemplate(L.template, vars);
    if (templated) return templated;

    const lines=[];
    lines.push(`${city}${date?`, le ${date}`:''}`);
    lines.push('');
    lines.push("Objet : Lettre d'accompagnement - Reglement par cheque");
    lines.push('');
    lines.push('Madame, Monsieur,');
    lines.push('');
    lines.push(`Veuillez trouver ci-joint le reglement par cheque n° ${chequeNo} au profit de ${payee}.`);
    lines.push(`Montant : ${amountNum} MAD (${amountWords}).`);
    lines.push('');
    lines.push('Factures reglees :');
    invoices.forEach(x=>lines.push(x));
    lines.push('');
    lines.push('Cordialement,');
    lines.push('');
    lines.push(L.company_name||'Entreprise');
    lines.push(L.signatory||'Signature');
    return lines;
  }

  function printCheque(){
    const api = getPdfApi();
    const blob = api.makeSinglePagePdfBlob({pageWmm:CHECK_W_MM, pageHmm:CHECK_H_MM, items: buildChequeItems()});
    api.openAndPrintPdf(blob);
  }

  function printLetter(){
    const api = getPdfApi();
    const lines = buildLetterLines();

    const items=[];
    let y=25;
    const line=6.5;

    items.push({x_mm:190, y_mm:y, size_pt:11, align:'right', text: lines[0]||''});
    y += line*2;
    items.push({x_mm:105, y_mm:y, size_pt:13, align:'center', text: lines[2]||''});
    y += line*2;

    const body = lines.slice(4);
    body.forEach(ln=>{
      if (y>270) return;
      items.push({x_mm:20, y_mm:y, size_pt:11, align:'left', text: ln||''});
      y += line;
      if (ln==='') y += line*0.6;
    });

    const blob = api.makeSinglePagePdfBlob({pageWmm:210, pageHmm:297, items});
    api.openAndPrintPdf(blob);
  }

  function renderPdfPreview(){
    const holder = $('#pdfHolder');
    if (!holder) return;
    try{
      const api = getPdfApi();
      const blob = api.makeSinglePagePdfBlob({pageWmm:CHECK_W_MM, pageHmm:CHECK_H_MM, items: buildChequeItems()});
      const url = URL.createObjectURL(blob);
      holder.innerHTML = `<iframe src="${url}" title="Aperçu PDF"></iframe>`;
    }catch(e){
      holder.innerHTML = `<div style="padding:14px;color:#94a3b8">PDF engine error: ${escapeHtml(e.message)}</div>`;
    }
  }

  function renderAllPreviews(){
    $$('.preview-root').forEach(r=>{
      renderPreview(r);
      enableDragging(r);
    });
  }

  function renderAll(){
    fillBankSelects();
    loadForm();
    renderSettingsGrid();
    renderAllPreviews();
    renderPdfPreview();
  }

  async function saveAll(){
    const r = await api('save', {config:state.config, banks:{banks:state.banks}});
    if (r.ok){ setDirty(false); toast('Enregistré'); }
    else toast(r.error||'Erreur sauvegarde');
  }

  function bindToolbar(){
    const s = $('#saveBtn'); if (s) s.onclick=()=>saveAll();
    const pc = $('#printChequeBtn'); if (pc) pc.onclick=()=>printCheque();
    const pl = $('#printLetterBtn'); if (pl) pl.onclick=()=>printLetter();
    const exp = $('#exportBtn'); if (exp) exp.onclick=()=>{ window.location.href='index.php?action=export'; };
    const imp = $('#importBtn');
    const impFile = $('#importFile');
    if (imp && impFile){
      imp.onclick=()=>impFile.click();
      impFile.onchange=async ()=>{
        const file = impFile.files?.[0]; if (!file) return;
        const text = await file.text();
        const r = await api('import', {text});
        if (r.ok){ location.reload(); } else toast(r.error||'Import invalide');
      };
    }

    const addB = $('#addBankBtn'); 
    if (addB) addB.onclick=async ()=>{
      const name = prompt('Nom de la banque ?'); if (!name) return;
      const r = await api('bank_add',{name});
      if (r.ok){ state.banks = r.banks.banks; state.currentBankId = r.new_id; renderAll(); setDirty(true); }
      else toast(r.error||'Erreur');
    };
    const delB = $('#delBankBtn');
    if (delB) delB.onclick=async ()=>{
      const bank=getBank(); if (!bank) return;
      if (!confirm(`Supprimer "${bank.name}" ?`)) return;
      const r = await api('bank_delete',{bank_id:bank.id});
      if (r.ok){ state.banks=r.banks.banks; state.currentBankId=state.banks[0]?.id||null; renderAll(); setDirty(true); }
      else toast(r.error||'Erreur');
    };
    const resetB = $('#resetBankBtn');
    if (resetB) resetB.onclick=async ()=>{
      const bank=getBank(); if (!bank) return;
      const r = await api('reset_bank',{bank_id:bank.id});
      if (r.ok){ state.banks=r.banks.banks; renderAll(); setDirty(true); }
      else toast(r.error||'Erreur');
    };

    const scan = $('#scanFile');
    if (scan){
      scan.onchange = async ()=>{
        const file = scan.files?.[0]; const bank=getBank();
        if (!file || !bank) return;
        const fd = new FormData();
        fd.append('scan', file);
        fd.append('bank_id', bank.id);
        const res = await fetch('index.php?action=upload_scan', {method:'POST', body:fd});
        const r = await res.json().catch(()=>({ok:false,error:'upload error'}));
        if (r.ok){ state.banks = r.banks.banks; renderAll(); setDirty(true); toast('Scan importé'); }
        else toast(r.error||'Erreur upload');
        scan.value='';
      };
    }

    const pp = $('#printPreviewBtn');
    if (pp) pp.onclick=()=>printCheque();
    const pl2 = $('#printLetterPageBtn');
    if (pl2) pl2.onclick=()=>printLetter();
  }

  async function boot(){
    const r = await api('load',{});
    if (!r.ok) { toast(r.error||'load error'); return; }
    state.user = r.user;
    state.banks = r.banks.banks || [];
    state.config = r.config || {};
    // guards for missing structures
    if (!state.config.check) state.config.check = {};
    if (!state.config.letter) state.config.letter = {company_name:'',signatory:'',template:''};
    state.currentBankId = state.config.last_bank_id || state.banks[0]?.id || null;
    bindToolbar();
    renderAll();
    bindInputs();
    setDirty(false);

    window.addEventListener('resize', ()=>renderAllPreviews());
  }

  document.addEventListener('DOMContentLoaded', ()=>{
    if (!$('#app')) return;
    boot().catch(e=>{ console.error(e); toast(e.message||'Erreur'); });
  });

})();
